/*   Copyright Anritsu 2002  All Rights Reserved.                            */
/*****************************************************************************/

#include <formatio.h>
#include <visa.h>
#include <vpptype.h>
#include <ansi_c.h>
#include <string.h>
#include <analysis.h>
#include "MT8820.h"


#define AUMT8820_MAINFRAME_REVISION     "Rev 3.1,01Sep2003, CVI 5.5"     /* Instrument driver revision */


/* = MT8820 Radio Communication Analyzer (VISA) ======= */
/* 
    LabWindows 5.5 Instrument Driver
    Original Release: September 2002
    Modification History:

        Sep 2002 -  Original version
        12 Oct 2002 -  Check in Japan
        10 Jan 2003 -  Added fp files
        09 Apr 2003 -  Changed Synch Sweep from TS;SWP? to SWP;SWP?
        24 Jun 2003 -  Check in Japan
        25 Jul 2003 - [V3.0], Anritsu MCA1, New Released 
        01 Sep 2003 - [V3.1], Anritsu MCA1, Deleted:INILEVEL?/MCAT?/MCFV?/MSTAT?

*/
/* ========================================================================= */

/*****************************************************************************/
/* 
	These first two functions are used to open/close an instrument session
 */
/*****************************************************************************/

/*****************************************************************************/
/* 
       Function:   Initialize
       Command(s): *RST,*IDN? if requested
       Purpose:    This function opens the instrument,and if requested, queries the instrument,
                     for its ID, and initializes the instrument to a known state.
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Init(ViRsrc resourceName, ViBoolean IDQuery, 
                                         ViBoolean resetInstr, ViSession *instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   ViSession rmSession = 0;
   
    /*- Check input parameter ranges ----------------------------------------*/
    if (AuMT8820_InvalidViBooleanRange(IDQuery))
        return VI_ERROR_INV_PARAMETER;
    if (AuMT8820_InvalidViBooleanRange(resetInstr))
        return VI_ERROR_INV_PARAMETER;

# ifndef NOHW
    /*- Open instrument session ---------------------------------------------*/
    if ((auMT8820_status = viOpenDefaultRM(&rmSession)) < 0)
        return auMT8820_status;

    if ((auMT8820_status = viOpen(rmSession, resourceName, VI_NULL, VI_NULL, instrSession)) < 0)
      {
        viClose (rmSession);
        return auMT8820_status;
       }

    /*- Configure VISA Formatted I/O ----------------------------------------*/
    if ((auMT8820_status = viSetAttribute(*instrSession, VI_ATTR_TMO_VALUE, 100000)) < 0)
        return AuMT8820_InitCleanUp(rmSession, instrSession, auMT8820_status);
    if ((auMT8820_status = viSetBuf(*instrSession, VI_READ_BUF|VI_WRITE_BUF, 4000)) < 0)
        return AuMT8820_InitCleanUp(rmSession, instrSession, auMT8820_status);
    if ((auMT8820_status = viSetAttribute(*instrSession, VI_ATTR_TERMCHAR, 10)) < 0)
        return AuMT8820_InitCleanUp(rmSession, instrSession, auMT8820_status);
    if ((auMT8820_status = viSetAttribute(*instrSession, VI_ATTR_TERMCHAR_EN, 1)) < 0)
        return AuMT8820_InitCleanUp(rmSession, instrSession, auMT8820_status);
    if ((auMT8820_status = viSetAttribute(*instrSession, VI_ATTR_WR_BUF_OPER_MODE, VI_FLUSH_ON_ACCESS)) < 0)
        return AuMT8820_InitCleanUp(rmSession, instrSession, auMT8820_status);
    if ((auMT8820_status = viSetAttribute(*instrSession, VI_ATTR_RD_BUF_OPER_MODE, VI_FLUSH_ON_ACCESS)) < 0)
        return AuMT8820_InitCleanUp(rmSession, instrSession, auMT8820_status);

    /*- Identification Query ------------------------------------------------*/
    if (IDQuery)
      {
        if ((auMT8820_status = AuMT8820_Write(*instrSession, "*IDN?", 5, &retCnt)) < 0)
            return AuMT8820_InitCleanUp(rmSession, instrSession, auMT8820_status);
        if ((auMT8820_status = AuMT8820_Read(*instrSession, rdBuf, BUFFER_SIZE, &retCnt)) < 0)
            return auMT8820_status;

        Scan (rdBuf, "ANRITSU,MT8820");
        if (NumFmtdBytes () !=14)
            return AuMT8820_InitCleanUp(rmSession, instrSession, VI_ERROR_FAIL_ID_QUERY);
      }

    /*- Reset instrument ----------------------------------------------------*/
    if (resetInstr)
     {
        if ((auMT8820_status = AuMT8820_Write(*instrSession, "*RST", 4, &retCnt)) < 0)
            return AuMT8820_InitCleanUp(rmSession, instrSession, auMT8820_status);
     }
    else  /*- Send Default Instrument Setup ---------------------------------*/
        if ((auMT8820_status = AuMT8820_DefaultInstrSetup(*instrSession)) < 0)
            return AuMT8820_InitCleanUp(rmSession, instrSession, auMT8820_status);
#endif
    return auMT8820_status;
}


/*****************************************************************************/
/*
       Function: Close
       Purpose:  This function closes the instrument
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Close(ViSession instrSession)
{
    ViSession rmSession;
    ViStatus  auMT8820_status = VI_SUCCESS;

    if ((auMT8820_status = viGetAttribute(instrSession, VI_ATTR_RM_SESSION, &rmSession)) < 0)
        return auMT8820_status;

    auMT8820_status = viClose(instrSession);
    viClose(rmSession);

    return auMT8820_status;
}


/*****************************************************************************/
/*									 MainFrame 								 */
/*			    The following functions are defined for Main Frame			 */
/*****************************************************************************/

/*****************************************************************************/
/*
       Function:   Set Synchronous Single Sweep
       Command(s): SWP[?],TS
       Purpose:    Start a synchronous single sweep, query is valid when using SWP to start a sweep
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Synchronous_Single_Sweep(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
 
   len = sprintf(buf, "SWP;SWP?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Synchronous Single Sweep
       Command(s): SWP[?],TS
       Purpose:    Start a synchronous single sweep, query is valid when using SNGLS to start a sweep
       Mode defined by: AUMT8820_SYNCHRONOUS_SINGLE_SWEEP_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Synchronous_Single_Sweep(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"COMPLETED", "EXECUTING", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SWP?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", mode);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Single Sweep Mode
       Command(s): SNGLS,S2
       Purpose:    Starts a single sweep, but any further commands stop sweep if not completed, use SWP to synchronize a sweep
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Single_Sweep_Mode(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "SNGLS");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Continuous Sweep Mode
       Command(s): CONTS,S1
       Purpose:    Starts continuous sweep mode, but any further commands stop sweep if not completed, use SWP to synchronize a sweep
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Continuous_Sweep_Mode(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "CONTS");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Stop Measurement
       Command(s): MEASSTOP
       Purpose:    Stops the current measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Stop_Measurement(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "MEASSTOP");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Identify Query
       Command(s): *IDN?
       Purpose:    Get the current instrument string
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Identify_Query(ViSession instrSession, ViChar _VI_FAR string[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "*IDN?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s",string);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Reset
       Command(s): *RST
       Purpose:    Reset the instrument
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Reset(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "*RST");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Operation Complete
       Command(s): *OPC[?]
       Purpose:    Operation Complete
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Operation_Complete(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "*OPC");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Operation Complete
       Command(s): *OPC[?]
       Purpose:    Operation Complete
       Mode defined by: AUMT8820_OPERATION_COMPLETE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Operation_Complete(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"COMPLETED", "EXECUTING", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "*OPC?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", mode);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Wait To Continue
       Command(s): *WAI
       Purpose:    Wait for command to complete
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Wait_To_Continue(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "*WAI");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Clear Status
       Command(s): *CLS
       Purpose:    Clear instrument status
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Clear_Status(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "*CLS");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Standard Event Status Enable
       Command(s): *ESE[?]
       Purpose:    Get/Set the standard event status mask
       Mask defined by:  AUMT8820_ESE_MASK_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Standard_Event_Status_Enable(ViSession instrSession, ViInt32 mask)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mask,0,255) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "*ESE %d", mask);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Standard Event Status Enable
       Command(s): *ESE[?]
       Purpose:    Get/Set the standard event status mask
       Mask defined by:  AUMT8820_ESE_MASK_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Standard_Event_Status_Enable(ViSession instrSession,  ViInt32 _VI_FAR *mask)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "*ESE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", mask);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Standard Event Status Register Query
       Command(s): *ESR?
       Purpose:    Get the current instrument standard event status register contents
       Value defined by:  AUMT8820_ESE_MASK_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Standard_Event_Status_Register_Query(ViSession instrSession, ViInt32 _VI_FAR *value)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "*ESR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", value);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Service Request Enable
       Command(s): *SRE[?]
       Purpose:    Get/Set the service request enable mask
       Mask defined by:  AUMT8820_STB_MASK_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Service_Request_Enable(ViSession instrSession, ViInt32 mask)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mask,0,255) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "*SRE %d", mask);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Service Request Enable
       Command(s): *SRE[?]
       Purpose:    Get/Set the service request enable mask
       Mask defined by:  AUMT8820_STB_MASK_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Service_Request_Enable(ViSession instrSession,  ViInt32 _VI_FAR *mask)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "*SRE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", mask);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Status Byte Query
       Command(s): *STB?
       Purpose:    Get the current instrument status byte register contents
       Value defined by:  AUMT8820_STB_MASK_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Status_Byte_Query(ViSession instrSession, ViInt32 _VI_FAR *value)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "*STB?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", value);

   return auMT8820_status;

}




/*****************************************************************************/
/*
       Function:   Revision
       Command(s): *IDN?
       Purpose:    Returns instrument firmware version if implemented, and driver version
       InstrRev[] form:  ANRITSU,MT8820A,product code, firmware revision
*/                                                                                                                                                      
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Revision(ViSession instrSession, ViChar _VI_FAR driverRev[], ViChar _VI_FAR instrRev[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "*IDN?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", instrRev);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Screen Mode
       Command(s): SCREEN[?]
       Purpose:    Get/Set the screen condition
       Mode defined by: AUMT8820_SCREEN_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Screen_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "SCREEN %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Trigger
       Command(s): *TRG
       Purpose:    Trigger a measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Trigger(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "*TRG");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status; 

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Self Test
       Command(s): *TST?
       Purpose:    Query the self test result
       Test_result defined by:  AUMT8820_TEST_RESULT_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Self_Test(ViSession instrSession, ViInt32 _VI_FAR *test_result)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "*TST?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d",test_result);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Alarm Sound Mode
       Command(s): ALARM[?],BEP
       Purpose:    Sets whether or not to sound the beep when an error occurs or for other conditions.
       Sound defined by: AUMT8820_ALARM_SOUND_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Alarm_Sound_Mode(ViSession instrSession, ViInt32 sound)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"OFF", "ON", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "ALARM %s", commandArray[sound]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Alarm Sound Mode
       Command(s): ALARM[?],BEP
       Purpose:    Gets whether or not to sound the beep when an error occurs or for other conditions.
       Sound defined by: AUMT8820_ALARM_SOUND_MODE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Alarm_Sound_Mode(ViSession instrSession, ViInt32  _VI_FAR *sound)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"OFF", "ON", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ALARM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, sound)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Sound Buzzer
       Command(s): BZR
       Purpose:    Sound the buzzer
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Sound_Buzzer(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "BZR");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Date
       Command(s): DATE[?]
       Purpose:    Change or query the instrument date (year,month,day)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Date(ViSession instrSession, ViInt32 year, ViInt32 month, ViInt32 day)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(year,2001,2030) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   if (AuMT8820_InvalidViInt32Range(month,1,12) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   if (AuMT8820_InvalidViInt32Range(day,1,31) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "DATE %d,%d,%d", year,month,day);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Date
       Command(s): DATE[?]
       Purpose:    Change or query the instrument date (year,month,day)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Date(ViSession instrSession,  ViInt32 _VI_FAR *year,  ViInt32 _VI_FAR *month,  ViInt32 _VI_FAR *day)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DATE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%d,%d", year, month, day);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Date Mode
       Command(s): DATEMODE[?]
       Purpose:    Sets the display format for the date
       Mode defined by: AUMT8820_DATE_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Date_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"YMD", "MDY", "DMY", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "DATEMODE %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Date Mode
       Command(s): DATEMODE[?]
       Purpose:    Gets the display format of the date
       Mode defined by: AUMT8820_DATE_MODE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Date_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"YMD", "MDY", "DMY", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DATEMODE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Panel Display Mode
       Command(s): DISPL[?]
       Purpose:    Setting for the power switch of the LCD display
       Mode defined by: AUMT8820_PANEL_DISPLAY_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Panel_Display_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "DISPL %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}

 
/*****************************************************************************/
/*
       Function:   Get Panel Display Mode
       Command(s): DISPL[?]
       Purpose:    Setting for the power switch of the LCD display
       Mode defined by: AUMT8820_PANEL_DISPLAY_MODE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Panel_Display_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DISPL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set END Event Status Enable Register
       Command(s): ESE2[?]
       Purpose:    Set/Get the End event status enable register
       SEregister defined by:  AUMT8820_ESE2_MASK_B0_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_END_Event_Status_Enable_Register(ViSession instrSession, ViInt32 SEregister)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(SEregister,0,255) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "ESE2 %d", SEregister);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get END Event Status Enable Register
       Command(s): ESE2[?]
       Purpose:    Set/Get the End event status enable register
       SEregister defined by:  AUMT8820_ESE2_MASK_B0_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_END_Event_Status_Enable_Register(ViSession instrSession,  ViInt32 _VI_FAR *SEregister)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ESE2?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", SEregister);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set ERR Event Status Enable Register
       Command(s): ESE3[?]
       Purpose:    Set/Get the Error event status enable register
       SEregister defined by:  AUMT8820_ESE3_MASK_B0_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_ERR_Event_Status_Enable_Register(ViSession instrSession, ViInt32 SEregister)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(SEregister,0,255) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "ESE3 %d", SEregister);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get ERR Event Status Enable Register
       Command(s): ESE3[?]
       Purpose:    Set/Get the Error event status enable register
       SEregister defined by:  AUMT8820_ESE3_MASK_B0_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_ERR_Event_Status_Enable_Register(ViSession instrSession,  ViInt32 _VI_FAR *SEregister)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ESE3?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", SEregister);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get END Event Status Register Query
       Command(s): ESR2?
       Purpose:    Query the end event status register
       Event_Status defined by:  AUMT8820_ESE2_MASK_B0_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_END_Event_Status_Register_Query(ViSession instrSession, ViInt32 _VI_FAR *event_Status)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ESR2?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", event_Status);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get ERR Event Status Register Query
       Command(s): ESR3?
       Purpose:    Query the error event status register
       Event_Status defined by:  AUMT8820_ESE3_MASK_B0_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_ERR_Event_Status_Register_Query(ViSession instrSession, ViInt32 _VI_FAR *event_Status)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ESR3?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", event_Status);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Reset To Default
       Command(s): FULLCAL_TM?
       Purpose:    Queries the time for the last Full Calibration
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Full_Calibration_Time(ViSession instrSession, ViInt32 _VI_FAR *year,ViInt32 _VI_FAR *month, ViInt32 _VI_FAR *day, ViInt32 _VI_FAR *hours, ViInt32 _VI_FAR *minutes, ViInt32 _VI_FAR *seconds)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "FULLCAL_TM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d,%d,%d,%d,%d,%d", year, month, day, hours, minutes, seconds);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Reset To Default
       Command(s): GTL
       Purpose:    Change the remote to local control
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Go_To_Local(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "GTL");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Reset To Default
       Command(s): INI
       Purpose:    Initialize all instrument set parameters to their system default values
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Reset_To_Default(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "INI");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Switching Count
       Command(s): MCAT?
       Purpose:    Query the set number for mechanical switches (18)
 */
/*****************************************************************************/
/*ViStatus _VI_FUNC AuMT8820_Get_Switching_Count(ViSession instrSession, ViChar _VI_FAR switching_Count_Status[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MCAT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", switching_Count_Status);

   return auMT8820_status;

}
*/

/*****************************************************************************/
/*
       Function:   Preset To System Initialization
       Command(s): PRE,IP
       Purpose:    Executes initialization at System Initial level
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Preset_To_System_Initialization(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "PRE");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Preset To Initialization Level
       Command(s): PRESET
       Purpose:    Executes initialization at specified initialization level
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Preset_To_Initialization_Level(ViSession instrSession, ViInt32 initialization_Level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString comandArray[] = {"SYSTEM", "INST", "FACTORY", "POWERON", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "PRESET %s", comandArray[initialization_Level]);

   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Parameters By Number
       Command(s): PRMRECALL
       Purpose:    Reads the parameter file and reflects the settings at the MT8820
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Parameters_By_Number(ViSession instrSession, ViInt32 file_Number)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(file_Number,0,999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "PRMRECALL %d", file_Number);		 
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Parameters By Next Number
       Command(s): PRMSAVE
       Purpose:    Saves the parameter file (smallest of those unused) and reflects the settings at the MT8820
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Parameters_By_Next_Number(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PRMSAVE");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Parameters By Number
       Command(s): PRMSAVE
       Purpose:    Saves the parameter file and reflects the settings at the MT8820
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Parameters_By_Number(ViSession instrSession,  ViInt32 file_Number)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(file_Number,0,999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "PRMSAVE %d", file_Number);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Parameters By Name
       Command(s): PRMRECALLNAME
       Purpose:    Reads the parameter file and reflects the settings at the MT8820
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Parameters_By_Name(ViSession instrSession,  ViChar _VI_FAR file_Name[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "PRMRECALLNAME \"%s\"", file_Name);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Parameters By Name
       Command(s): PRMSAVENAME
       Purpose:    Saves the parameter file and reflects the settings at the MT8820
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Parameters_By_Name(ViSession instrSession, ViChar _VI_FAR file_Name[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PRMSAVENAME \"%s\"", file_Name);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Reference Frequency
       Command(s): REF[?]
       Purpose:    Set/Get the reference input frequency
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Reference_Frequency(ViSession instrSession, ViInt32 frequency)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"10MHZINT", "10MHZEXT", "13MHZEXT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "REF %s", commandArray[frequency]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Reference Frequency
       Command(s): REF[?]
       Purpose:    Set/Get the reference input frequency
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Reference_Frequency(ViSession instrSession, ViInt32  _VI_FAR *frequency)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"10MHZINT", "10MHZEXT", "13MHZEXT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "REF?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, frequency)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set RF Siganl Output Connector
       Command(s): RFOUT[?]
       Purpose:    Set/Get the RF signal output connector
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_RF_Signal_Output_Connector(ViSession instrSession, ViInt32 connector)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"MAIN", "AUX", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "RFOUT %s", commandArray[connector]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Siganl Output Connector
       Command(s): RFOUT[?]
       Purpose:    Set/Get the RF signal output connector
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Signal_Output_Connector(ViSession instrSession, ViInt32  _VI_FAR *connector)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"MAIN", "AUX", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "RFOUT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, connector)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Parameter Directory Query
       Command(s): RLCD?
       Purpose:    Query the absolute path to the directory in which the parameter files will be accessed
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Parameter_Directory_Query(ViSession instrSession, ViChar _VI_FAR directory[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "RLCD?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", directory);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Parameter Read Directory List
       Command(s): RLFLIST?
       Purpose:    Query the parameter file names and attributes in the current directory
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Parameter_Read_Directory_List(ViSession instrSession, ViChar _VI_FAR read_Files[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "RLFLIST?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", read_Files);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Serial Number Query
       Command(s): SERIAL?
       Purpose:    Query the serial number of the MT8820
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Serial_Number_Query(ViSession instrSession, ViChar _VI_FAR serial_Number[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SERIAL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", serial_Number);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Standard Select
       Command(s): STDSEL[?]
       Purpose:    Set/Get the modulation standard for the MT8820
       Standard defined by:  AUMT8820_STANDARD_SELECT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Standard_Select(ViSession instrSession, ViInt32 standard)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"WCDMA", "GSM", "CDMA2K", "PDC", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "STDSEL %s", commandArray[standard]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Standard Select
       Command(s): STDSEL[?]
       Purpose:    Set/Get the modulation standard for the MT8820
       Standard defined by:  AUMT8820_STANDARD_SELECT_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Standard_Select(ViSession instrSession, ViInt32  _VI_FAR *standard)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"WCDMA", "GSM", "CDMA2K", "PDC", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "STDSEL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, standard)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Parameter Save Directory Query
       Command(s): SVCD?
       Purpose:    Query the path to the parameter saving directory
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Parameter_Save_Directory_Query(ViSession instrSession, ViChar _VI_FAR save_Directory[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SVCD?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", save_Directory);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Parameter Default Filename Query
       Command(s): SVDEFFILE?
       Purpose:    Query the file name to be automatically created when saving the setting parameters
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Parameter_Default_Filename_Query(ViSession instrSession, ViChar _VI_FAR default_Name[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SVDEFFILE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", default_Name);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Parameter Save Directory List
       Command(s): SVFLIST?
       Purpose:    Query the parameter saving directory file names and attributes
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Parameter_Save_Directory_List(ViSession instrSession, ViChar _VI_FAR file_List[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SVFLIST?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", file_List);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Time
       Command(s): TIME[?]
       Purpose:    Set/Get the current instrument time
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Time(ViSession instrSession,  ViInt32 hours,  ViInt32 minutes,  ViInt32 seconds)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(hours,0,23) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   if (AuMT8820_InvalidViInt32Range(minutes,0,59) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   if (AuMT8820_InvalidViInt32Range(seconds,0,59) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
      
   len = sprintf(buf, "TIME %d,%d,%d", hours, minutes, seconds);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Time
       Command(s): TIME[?]
       Purpose:    Set/Get the current instrument time
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Time(ViSession instrSession, ViInt32 _VI_FAR *hours, ViInt32 _VI_FAR *minutes, ViInt32 _VI_FAR *seconds)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TIME?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d,%d,%d" , hours, minutes, seconds);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set User Title
       Command(s): TITLE[?]
       Purpose:    Set/Get the user-defined character string displayed in the title
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_User_Title(ViSession instrSession,  ViChar _VI_FAR title[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "TITLE \"%s\"", title);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get User Title
       Command(s): TITLE[?]
       Purpose:    Set/Get the user-defined character string displayed in the title
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_User_Title(ViSession instrSession, ViChar _VI_FAR title[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TITLE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s" ,title);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power On Time
       Command(s): TMCNT?
       Purpose:    Query the accumulated power-on time in minutes (resolution 1 min)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_On_Time(ViSession instrSession, ViInt32 _VI_FAR *accumulated_Time)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TMCNT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%d", accumulated_Time);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set GPIB Terminator
       Command(s): TRM[?]
       Purpose:    Set/Get the GPIB termination character
       Terminator defined by:  AUMT8820_GPIB_TERMINATOR_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_GPIB_Terminator(ViSession instrSession, ViInt32 terminator)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"0", "1", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "TRM %s", commandArray[terminator]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get GPIB Terminator
       Command(s): TRM[?]
       Purpose:    Set/Get the GPIB termination character
       Terminator defined by:  AUMT8820_GPIB_TERMINATOR_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_GPIB_Terminator(ViSession instrSession, ViInt32  _VI_FAR *terminator)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"0", "1", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TRM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, terminator)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Title Display Type
       Command(s): TTL[?]
       Purpose:    Set/Get the type of information to be displayed in the title display field
       Type defined by:  AUMT8820_TITLE_DISPLAY_TYPE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Title_Display_Type(ViSession instrSession, ViInt32 type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"USER", "DATE", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "TTL %s", commandArray[type]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Title Display Type
       Command(s): TTL[?]
       Purpose:    Set/Get the type of information to be displayed in the title display field
       Type defined by:  AUMT8820_TITLE_DISPLAY_TYPE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Title_Display_Type(ViSession instrSession, ViInt32  _VI_FAR *type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"USER", "DATE", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%s", resBuf0); 

   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, type)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*= UTILITY ROUTINES (Non-Exportable Functions) =============================*/
/*****************************************************************************/

/*****************************************************************************/
/* Function: Boolean Value Out Of Range - ViBoolean                          */
/* Purpose:  This function checks a Boolean to see if it is equal to VI_TRUE */
/*           or VI_FALSE. If the value is out of range, the return value is  */
/*           VI_TRUE, otherwise the return value is VI_FALSE.                */
/*****************************************************************************/
ViBoolean AuMT8820_InvalidViBooleanRange(ViBoolean val)
{
    return ((val != VI_FALSE && val != VI_TRUE) ? VI_TRUE : VI_FALSE);
}

/*****************************************************************************/
/* Function: Short Signed Integer Value Out Of Range - ViInt8                */
/* Purpose:  This function checks a short signed integer value to see if it  */
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*****************************************************************************/
ViBoolean AuMT8820_InvalidViInt8Range(ViInt8 val, ViInt8 min, ViInt8 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*****************************************************************************/
/* Function: Signed Integer Value Out Of Range - ViInt16                     */
/* Purpose:  This function checks a signed integer value to see if it        */
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*****************************************************************************/
ViBoolean AuMT8820_InvalidViInt16Range(ViInt16 val, ViInt16 min, ViInt16 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*****************************************************************************/
/* Function: Long Signed Integer Value Out Of Range - ViInt32                */
/* Purpose:  This function checks a long signed integer value to see if it   */
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*****************************************************************************/
ViBoolean AuMT8820_InvalidViInt32Range(ViInt32 val, ViInt32 min, ViInt32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*****************************************************************************/
/* Function: Short Unsigned Integer Value Out Of Range - ViUInt8             */
/* Purpose:  This function checks a short unsigned integer value to see if it*/
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*****************************************************************************/
ViBoolean AuMT8820_InvalidViUInt8Range(ViUInt8 val, ViUInt8 min, ViUInt8 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*****************************************************************************/
/* Function: Unsigned Integer Value Out Of Range - ViUInt16                  */
/* Purpose:  This function checks a  unsigned integer value to see if it     */
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*****************************************************************************/
ViBoolean AuMT8820_InvalidViUInt16Range(ViUInt16 val, ViUInt16 min, ViUInt16 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*****************************************************************************/
/* Function: Long Unsigned Integer Value Out Of Range - ViUInt32             */
/* Purpose:  This function checks a long unsigned integer value to see if it */
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*****************************************************************************/
ViBoolean AuMT8820_InvalidViUInt32Range(ViUInt32 val, ViUInt32 min, ViUInt32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*****************************************************************************/
/* Function: Real (Float) Value Out Of Range - ViReal32                      */
/* Purpose:  This function checks a real (float) value to see if it lies     */
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*****************************************************************************/
ViBoolean AuMT8820_InvalidViReal32Range(ViReal32 val, ViReal32 min, ViReal32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*****************************************************************************/
/* Function: Real (Double) Value Out Of Range - ViReal64                     */
/* Purpose:  This function checks a real (ViReal64) value to see if it lies  */
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*****************************************************************************/
ViBoolean AuMT8820_InvalidViReal64Range(ViReal64 val, ViReal64 min, ViReal64 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*****************************************************************************/
/* Function: Initialize Clean Up                                             */
/* Purpose:  This function is used only by the MT8820_init function.  When   */
/*           an error is detected this function is called to close the       */
/*           open resource manager and instrument object sessions and to     */
/*           set the instrSession that is returned from MT8820_init to       */
/*           VI_NULL.                                                        */
/*****************************************************************************/
ViStatus AuMT8820_InitCleanUp(ViSession openRMSession,
                    ViPSession instrSession, ViStatus currentStatus)
{
    viClose(*instrSession);
    viClose(openRMSession);
    *instrSession = VI_NULL;

    return currentStatus;
}


/*****************************************************************************/
/*----------- INSERT INSTRUMENT-DEPENDENT UTILITY ROUTINES HERE -------------*/
/*****************************************************************************/

/*****************************************************************************/
/* Function: Write to MT8820                                                 */
/* Purpose:  This function formats a string and then sends it through VISA to*/
/*           the instrument. Because the instrument requires a line feed as a*/
/*           termination character, this routine adds one to the string.     */
/*           This function can be used to allow the driver to be debugged on */
/*           a computer that is not connected to a physical instrument by    */
/*           skipping the VISA call.                                         */
/*****************************************************************************/

ViStatus AuMT8820_Write(ViSession instrSession, ViChar *buf, ViUInt32 cnt, ViUInt32 *retCnt)
{
    ViStatus auMT8820_status = VI_SUCCESS;
   	ViChar delimitBuf[BUFFER_SIZE+1];

   	strcpy (delimitBuf, buf);
   	strcat (delimitBuf, "\n");
   	cnt++;

# ifndef NOHW

    auMT8820_status = viWrite(instrSession, delimitBuf, cnt, retCnt);
# else

	auMT8820_status = MessagePopup("Instrument command", delimitBuf);
#endif

    return auMT8820_status;
}

/*****************************************************************************/
/* Function: Read from MT8820          .                                     */
/* Purpose:  Reads a response from the MT8820 and places it in a null        */
/*           terminated string.                                              */
/*           ViStatus is returned which contains a VISA error code or		 */
/*           VI_SUCCESS if the function had no errors                        */
/*****************************************************************************/

ViStatus AuMT8820_Read(ViSession instrSession, ViChar *buf, ViUInt32 cnt, ViUInt32 *retCnt)
{
    ViStatus auMT8820_status = VI_SUCCESS;

# ifndef NOHW
    if (auMT8820_status = viRead( instrSession, buf, cnt, retCnt) <0)
    	return auMT8820_status;
    buf[(*retCnt) - 1] = 0;  /*  Null terminate the buffer. viRead does not */
                            /*  always do this. Also removes the LF character */
                            /*  from the end of the string. */

# else

   auMT8820_status = PromptPopup("Instrument read",
					 "type in the response expected from the instrument",
					 buf, 250);
# endif

    return auMT8820_status;

}

/*****************************************************************************/
/* Function: String to enum            .                                     */
/* Purpose:  Searches an array of strings for an element which matches a     */
/*           target string.                                                  */
/*           Index of matching element is returned.                          */
/* Notes:                                                                    */
/* 1. Any target string beginning and ending quotes are ignored.			 */
/* 2. The array of strings MUST be terminated with a null pointer, e.g.,     */
/*        char* myArray[] = {"string1","string2", ...., 0L};                 */
/*    If the array is not terminated with a null pointer, and the target     */
/*    string does not match any elements in the array, the search will run   */
/*    off the end of the array, probably causing a GENERAL PROTECTION FAULT. */
/*****************************************************************************/

ViStatus AuMT8820_StringToEnum(ViChar **enumStrings, ViChar *targetString, ViUInt32 *index)
{
	ViChar **scanPtr, *endPointer;
 ViUInt16 result=0;
	*index = 0;
	scanPtr = enumStrings;
	if ((targetString!= 0L) && (strlen(targetString) > 0))
	{
		/* Strip target string beginning and ending quotes.. */
		if (*targetString == '\"') targetString++;
		endPointer = targetString + strlen(targetString) - 1;
		if (*endPointer == '\"') *endPointer ='\0';

		/* Look for a match with the expected strings. */
		while (*scanPtr != 0L)
		{
			if(strcmp(targetString,*scanPtr)==0)
			{
				*index = result;
				return VI_SUCCESS;
			}
			scanPtr++; result++;
		}
	}
	return VI_ERROR_INV_RESPONSE;
}


/*****************************************************************************/
/* Function: Coerce float to step size                                       */
/* Purpose:  This function takes a floating point value in and makes it      */
/*           exactly divisable by the step value.                            */
/*           The function returns the value after coercion.                  */
/*           The value is always rounded down to the next allowable value.   */
/*                                                                           */
/*           For example if 												 */
/*                                                                           */
/*			 valueToCoerce = 12.3											 */
/*           stepSize = 0.25												 */
/*           the return value =12.25                                         */
/*                                                                           */
/*****************************************************************************/

ViReal64 AuMT8820_CoerceFloatToStep(ViReal64 valueToCoerce, ViReal64 stepSize)
{

	 ViReal64 divResult;

	 divResult = floor(valueToCoerce/stepSize)*stepSize;

	return divResult;
}

/*****************************************************************************/
/* Function: Coerce Int to step size                                         */
/* Purpose:  This function takes a Integer value in and makes it             */
/*           exactly divisable by the step value.                            */
/*           The function returns the value after coercion.                  */
/*           The value is always rounded down to the next allowable value.   */
/*                                                                           */
/*           For example if 												 */
/*                                                                           */
/*			 valueToCoerce = 1790											 */
/*           stepSize = 1000												 */
/*           the return value =1000                                          */
/*                                                                           */
/*****************************************************************************/

ViUInt32 AuMT8820_CoerceIntToStep(ViUInt32 valueToCoerce, ViUInt32 stepSize)
{

	 ViUInt32 divResult;

	 divResult = floor(valueToCoerce/stepSize)*stepSize;

	return divResult;
}


/*****************************************************************************/
/* Function: Default Instrument Setup                                        */
/* Purpose:  This function sends a default setup to the instrument.  This    */
/*           function is called by the MT8820_reset operation and by the     */
/*           MT8820_init function if the reset option has not been           */
/*           selected.  This function is useful for configuring any          */
/*           instrument settings that are required by the rest of the        */
/*           instrument driver functions such as turning headers ON or OFF   */
/*           or using the long or short form for commands, queries, and data.*/
/*****************************************************************************/
ViStatus AuMT8820_DefaultInstrSetup(ViSession instrSession)
{
    ViStatus auMT8820_status = VI_SUCCESS;
    ViUInt32 retCnt = 0;

    return auMT8820_status;
}

/*****************************************************************************/
/* Function: Retrieve a variable number of quoted strings.                   */
/* Purpose:  This function scans a buffer containing a variable number of    */
/*           comma-separated quoted strings, for example, 					 */
/*           	"STR1","STR2","STR3",..."STRN"                               */
/*           The strings are placed in a string array.                       */
/*  The user of the driver must supply the string array as a null terminated */
/*  list of destination string pointers, such as:                            */
/*  	ViChar* destString1[256], destString2[256], destString3[256],...;    */
/*  	ViChar** destStrings = {destString1,destString2,destString3,...,0L}; */
/*  This procedure will put a null string in the first unused 'destStringx'. */
/*****************************************************************************/
ViStatus AuMT8820_ParseStrings(ViChar *buffer, ViChar **destStrings)
{
    ViStatus auMT8820_status = VI_SUCCESS;
    ViUInt32 bytesRead=strlen(buffer),bytesScanned=0;
    ViChar** destScanner = destStrings;

    if ((buffer != 0L) && (strlen(buffer) >0) && (destStrings != 0L))
    {
    	while (bytesScanned < bytesRead)
    	{
    		if (*destScanner != 0L)
    		{
    			sscanf(buffer+bytesScanned, "%256[^,],", *destScanner);
	    		bytesScanned+= strlen(*destScanner)+1;
    			destScanner++;
    		}
    		else
    		{
    		auMT8820_status = VI_ERROR_INV_PARAMETER;
    			break;
    		}
    	}
    	/*  Flag the end of the returned addresses.(Not an error if the string array is exactly filled. */
    	if (*destScanner != 0L) strcpy(*destScanner,"");
    }
    else auMT8820_status = VI_ERROR_INV_PARAMETER;

    return auMT8820_status;

}

/*****************************************************************************/
/* Function: Convert an integer to a zero-fill binary string                 */
/* Purpose:  This function converts an integer parameter to an arbitrary     */
/*           length string of character ones and zeros.                      */
/*           Up to thirty-two bytes will be meaningful, but the left-fill    */
/*           can continue indefinitely.  An input 'stringSize' parameter     */
/*           determines how many bytes to fill.                              */
/*           Example and input integer of 255 and a size of 17 will produce  */
/*           the following string:											 */
/*           	"0000000011111111"                                           */
/* Notes:                                                                    */
/*  The user must count the null termination within the string allocation.   */
/*  There is no concept of a sign bit.                                       */
/*****************************************************************************/

void AuMT8820_IntegerToAsciiBitString(ViUInt32 integer, ViChar *asciiBitString, ViUInt32 stringSize)
{
	ViChar* stringScanner = asciiBitString + stringSize-1;
	*stringScanner = '\0';
	while (stringScanner > asciiBitString)
	{
		*(--stringScanner) = (integer & 1)?'1':'0';
		integer /= 2;
	}
}

/*****************************************************************************/
/* Function: Convert a zero-fill binary string to an integer                 */
/* Purpose:  This function converts an input string of ones and zeros to an  */
/*           integer.														 */
/*           Up to thirty-two bytes can be handled.                          */
/*           Example: an input string:                                       */
/*           	"0000000011111111"                                           */
/*           will return a value of 255.                                     */
/* Notes:                                                                    */
/*  The bit string must be right-justified within the parameter string.      */
/*  There is no concept of a sign bit.                                       */
/*  Any character other than zero or one will stop the right-to-left scan.   */
/*****************************************************************************/

ViUInt32 AuMT8820_AsciiBitStringToInteger(ViChar *asciiBitString)
{
	ViChar* stringScanner = asciiBitString + strlen(asciiBitString);
	ViChar myChar;
	ViUInt32 integer, myBitValue = 1;

	integer = 0;
	while (stringScanner > asciiBitString)
	{
		myChar = *(--stringScanner);
		/* If we encounter anything other than ones and zeroes, exit. */
		if ((myChar != '1') && (myChar != '0')) break;
		/* If we have a '1', increase the value of the returned integer. */
		if (myChar == '1') integer += myBitValue;
		/* Remember what power of two we are looking at. */
		myBitValue *= 2;
	}

	return(integer);
}

/*****************************************************************************/
/*=== END INSTRUMENT DRIVER SOURCE CODE =====================================*/
/*****************************************************************************/
